package lrs.lazyLRSEvaluators;

import lrs.*;
import fp.*;

/**
 * A "generator" function that contains the power and coefficient of a 
 * term in a polynomial expansion of a function.
 * Also has the ability to advance to the next term of the expansion.
 */
abstract class APolyGenerator {
  public double power = 0.0; 
  public double coef = 1.0;
  
  /**
   * Advances the power and coef to the next term in the expansion
   */
  public abstract void nextTerm();
}

/**
 * Generator for e^x
 */
class EGenerator extends APolyGenerator {
  public void nextTerm() {
    power++;
    coef *= 1.0/power;
  }
}

/**
 * Generator for sin(x) 
 * Note that x is in radians, so (to within round-off errors)
 * sin(0) = 0
 * sin(1.5707963267948966192313216916398) = 1
 * sin(-1.5707963267948966192313216916398) = -1
 * sin(3.1415926535897932384626433832795) = 0
 */
class SineGenerator extends APolyGenerator {
  private int n = 0;
  
  public SineGenerator() {
    power = 2*n+1;
  }
  
  public void nextTerm() {
    n++;
    power = 2*n+1;
    coef /= -2*n*(2*n+1);
  }
}

/**
 * Generator for cos(x) 
 * Note that x is in radians, so (to within round-off errors)
 * cos(0) = 1
 * cos(1.5707963267948966192313216916398) = 0
 * cos(-1.5707963267948966192313216916398) = 0
 * cos(3.1415926535897932384626433832795) = -1
 */
class CosineGenerator extends APolyGenerator {
  private int n = 0;
  
  public CosineGenerator() {
    power = 2*n;
  }
  
  public void nextTerm() {
    n++;
    power = 2*n;
    coef /= -2*n*(2*n-1);
  }
}

public class LazyLambdaEval extends ALazyEval
{
//  private APolyGenerator gen = new EGenerator();
  private APolyGenerator gen = new SineGenerator();
//  private APolyGenerator gen = new CosineGenerator();
  
  public final LRStruct nextLRS()
  {
    gen.nextTerm();
    return makeLRS();
  }
  
  public final LRStruct makeLRS()
  {
    return makeLazyLRS ( new PowerFn(gen.coef, gen.power));
  }
}

